<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\KAI\AcUnit;
use Illuminate\Http\Request;

class AcUnitController extends Controller
{
    // GET /api/stasiun/ac-units
    public function index(Request $request)
    {
        $query = AcUnit::with('station:id,name');
        
        // GET /api/stasiun/ac-units?station_id=1
        if ($request->has('station_id')) {
            $query->where('station_id', $request->station_id);
        }
        return $query->get();
    }

    // POST /api/stasiun/ac-units
    public function store(Request $request)
    {
        $validated = $request->validate([
            'station_id' => 'required|exists:stations,id',
            'carriage_number' => 'required|string|max:50',
            'unit_number' => 'required|string|max:50',
            'location' => 'nullable|string|max:100',
        ]);
        $acUnit = AcUnit::create($validated);
        return response()->json($acUnit, 201);
    }

    // GET /api/stasiun/ac-units/{id}
    public function show(AcUnit $acUnit)
    {
        return $acUnit->load('station:id,name');
    }

    // PUT /api/stasiun/ac-units/{id}
    public function update(Request $request, AcUnit $acUnit)
    {
        $validated = $request->validate([
            'station_id' => 'exists:stations,id',
            'carriage_number' => 'string|max:50',
            'unit_number' => 'string|max:50',
            'location' => 'nullable|string|max:100',
        ]);
        $acUnit->update($validated);
        return response()->json($acUnit);
    }

    // DELETE /api/stasiun/ac-units/{id}
    public function destroy(AcUnit $acUnit)
    {
        $acUnit->delete();
        return response()->json(null, 204);
    }
    
    // --- FITUR KONTROL ---

    // POST /api/stasiun/ac-units/{id}/control
    public function controlUnit(Request $request, $id)
    {
        $validated = $request->validate([
            'is_on' => 'nullable|boolean',
            'mode' => 'nullable|string|in:auto,cool,fan',
            'setpoint' => 'nullable|numeric|min:16|max:30',
        ]);
        
        $acUnit = AcUnit::findOrFail($id);
        
        // Update database
        $acUnit->update($validated);
        
        // (Di implementasi nyata, di sini Anda kirim perintah ke IoT)
        
        return response()->json($acUnit);
    }

    // POST /api/stasiun/ac-units/group-control
    public function groupControl(Request $request)
    {
        $validated = $request->validate([
            'station_id' => 'required|exists:stations,id',
            'is_on' => 'nullable|boolean',
            // ... (bisa tambah 'mode' atau 'setpoint' jika mau)
        ]);

        $stationId = $validated['station_id'];
        unset($validated['station_id']);

        $affectedRows = AcUnit::where('station_id', $stationId)->update($validated);

        return response()->json([
            'message' => 'Perintah kontrol grup terkirim.',
            'unit_terdampak' => $affectedRows
        ]);
    }
}