<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\KAI\SensorReading;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SensorReadingController extends Controller
{
    
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'ac_unit_id' => 'required|integer|exists:ac_units,id',
            'temperature' => 'nullable|numeric',
            'power' => 'nullable|numeric',
            'current' => 'nullable|numeric',
            'voltage' => 'nullable|numeric',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'recorded_at' => 'nullable|date',
        ]);

        if (!isset($validatedData['recorded_at'])) {
            $validatedData['recorded_at'] = now();
        }
        
        // Langsung simpan ke DB
        $reading = SensorReading::create($validatedData);

        return response()->json($reading, 201); // 201 = Created
    }

    /**
     * GET /api/stasiun/latest-readings
     * (GET Data Terakhir untuk Dashboard)
     */
    public function getLatestReadings()
    {
        $latestReadings = SensorReading::with('acUnit:id,carriage_number,unit_number')
            ->whereIn('id', function ($query) {
                $query->select(DB::raw('MAX(id)'))
                      ->from('sensor_readings')
                      ->groupBy('ac_unit_id');
            })
            ->orderBy('recorded_at', 'desc')
            ->get();

        return response()->json($latestReadings);
    }
    
    /**
     * GET /api/stasiun/sensor-history/{ac_unit_id}
     * (GET Data History untuk Line Chart)
     */
    public function getSensorHistory(Request $request, $ac_unit_id)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $query = SensorReading::where('ac_unit_id', $ac_unit_id);

        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('recorded_at', [$request->start_date, $request->end_date]);
        }

        $history = $query->orderBy('recorded_at', 'asc')->get();
        return response()->json($history);
    }
    
    /**
     * GET /api/stasiun/sensor-table
     * (GET Data Tabular)
     */
    public function getSensorTable(Request $request)
    {
        $data = SensorReading::with('acUnit:id,carriage_number,unit_number')
                    ->orderBy('recorded_at', 'desc')
                    ->paginate(50); // 50 data per halaman
        
        return response()->json($data);
    }
}